using System;
using System.Collections.Generic;
using System.Text;
using EzeScan.Interfaces;

namespace EDRMS_Output
{
    class Connection_Class: IConnection
    {
        static internal Helper PluginHelper;
        static internal ILogging DebugLog;
        static internal String StorageFolder;
        static internal String UserName;
        static internal String PreviousDocumentReference;
        static internal Boolean isConnected;
        static internal ISystemSettings SystemSettings;

        private Output_Class _outputHandler;
        private UserConfigurableParameter[] _connectionParameters;

        static internal void DebugError(Exception ex)
        {

            if ((DebugLog != null) && (ex != null))
            {
                String message = String.Format("{0}: {1}: {2}", ex.Message, ex.TargetSite, ex.StackTrace);
                DebugLog.LogPluginMessage( message, EzeScan.Interfaces.PluginDebugLevel.ErrorSevere);
            }
        }

        static internal void DebugError(String message)
        {

            if (DebugLog != null)
            {
                if (message == null)
                {
                    message = "";
                }

                DebugLog.LogPluginMessage(message, EzeScan.Interfaces.PluginDebugLevel.ErrorSevere);
            }

        }


        public Connection_Class()
        {
            isConnected = false;
            PreviousDocumentReference = "";

            // Initialise the shared objects required by the sample plugin
            PluginHelper = new EzeScan.Interfaces.Helper();
            _outputHandler = new Output_Class();
    
            // Create the connection parameters
            UserConfigurableParameter new_parameter = PluginHelper.CreateParameter("user", "User", ParameterType.String, "Bob", "Enter your username (hint: try Bob)", false);
            PluginHelper.AddParameter(ref _connectionParameters, ref new_parameter);

            new_parameter = PluginHelper.CreateParameter("pass", "Password", ParameterType.Password, "bob", "Enter your password (hint: try bob)", false);
            PluginHelper.AddParameter(ref _connectionParameters, ref new_parameter);

            new_parameter = PluginHelper.CreateParameter("folder", "Storage Folder", ParameterType.FolderPath, "C:\\Storage", "Enter your Storage Folder where documents will be stored", false);
            PluginHelper.AddParameter(ref _connectionParameters, ref new_parameter);            
        }

        #region IConnection Members

        bool IConnection.Initialise(ref ISystemSettings system_settings, ref ILogging plugin_log)
        {
            // EzeScan calls this function mainly to pass the following 2 helper objects
            // It also allows the plugin to do any extra setup required before a connection occurs

            // EzeScan passes a log object to the plugin
            DebugLog = plugin_log;

            // EzeScan passes a system settings object to the plugin
            SystemSettings = system_settings;

            return true;
        }

        UserConfigurableParameter[] IConnection.ConnectionParameters
        {
            get
            {
                // EzeScan calls this property to retrieve an array of parameters that are required to establish a connection
                // (e.g a Database connection may require parameters for "Server", "User", "Password", "Database", etc)

                return _connectionParameters;
            }
        }

        bool IConnection.Connect(ref UserConfigurableParameter[] connection_parameters)
        {
            // EzeScan calls this function before it performs any operation that requires or has requested a connection
    
            // The connection_parameters array is the same one passed out via IConnection_Properties, however now the "Value" property of each element has been filled in
            
            try
            {
                
                String user_name = "";
                String user_password = "";
                String folder_path = "";                                
                    
                // Process the parameters
                
                for (Int32 param_index = 0; param_index < PluginHelper.ParameterCount(ref connection_parameters); param_index++)
                {
                    
                    switch (connection_parameters[param_index].ID)
                    {
                        case "user":
                            user_name = connection_parameters[param_index].Value;
                            break;
                        case "pass":
                            user_password = connection_parameters[param_index].Value;
                            break;
                        case "folder":
                            folder_path = connection_parameters[param_index].Value;
                            break;
                    }
                    
                }
                
                if (isConnected && (StorageFolder == folder_path))
                {
                    DebugLog.LogPluginMessage("Using Existing Connection", PluginDebugLevel.InfoNormal);
                }
                else
                {
                    DebugLog.LogPluginMessage("Connecting", PluginDebugLevel.InfoNormal);
                    
                    // Process the login information
                    
                    if (user_name == "Bob")
                    {
                        DebugLog.LogPluginMessage("Using user: " + user_name, PluginDebugLevel.InfoNormal);
                        
                        if (user_password == "bob")
                        {
                            DebugLog.LogPluginMessage("Valid Password", PluginDebugLevel.InfoNormal);
                            
                            if (System.IO.Directory.Exists(folder_path))
                            {
                                DebugLog.LogPluginMessage("Using folder: " + folder_path, PluginDebugLevel.InfoNormal);
                                isConnected = true;
                                StorageFolder = folder_path;
                                UserName = user_name;
                            }
                            else
                            {
                                DebugLog.LogUserErrorMessage("Invalid Folder Path: " + folder_path);
                            }
                        }   
                        else
                        {
                            DebugLog.LogUserErrorMessage("Invalid User or Password");
                        }
                    
                    }
                    else
                    {
                        DebugLog.LogUserErrorMessage("Invalid User or Password");
                    }

                }
                
                return isConnected;

            }
            catch (Exception ex)
            {
                DebugError(ex);
                return false;
            }
        }

        SpeedType IConnection.ConnectionSpeed
        {
            get 
            { 
                // Use SpeedType_Fast for connections that *fail* quickly (e.g don't time out for more than 1 or 2 seconds)
                return SpeedType.Fast;
    
                // Otherwise use SpeedType.Slow 
            }
        }

        void IConnection.Disconnect(bool full_disconnect)
        {
            // EzeScan calls this function after it performs any operation that requires or has requested a connection
    
            // There are 2 types of disconnect, allowing plugins greater flexibility in managing their connections
            // * A partial disconnect (full_disconnect = False) is to notify the plugin that EzeScan has temporarily finished with the connection (however may require to use it again shortly)
            // * A full disconnect (full_disconnect = True) is passed when EzeScan has finished with the connection (e.g the application is about to close or a job using a different plugin or EDRMS is selected)
            
            if (full_disconnect && isConnected)
            {
                DebugLog.LogPluginMessage( "Disconnecting", PluginDebugLevel.InfoNormal);
                isConnected = false;
                StorageFolder = "";
            }
            else
            {
                if (isConnected)
                {
                    DebugLog.LogPluginMessage("Keeping existing connection", PluginDebugLevel.InfoNormal);
                }
                else
                {
                    DebugLog.LogPluginMessage("not currently connected", PluginDebugLevel.InfoNormal);
                }            
            }
        }

        int IConnection.RecentDocumentProperty(int property_id, ref string value_out)
        {
            // EzeScan calls this function after a document has been Imported, Outputted or Uploaded to retrieve details that can be used to:
            // * Display a confirmation to the user
            // * Output or Upload a new version of an existing document
            // * Print a barcode containing document information
            
            Int32 return_code = 1; // supported
            
            switch (property_id)
            {
                case 0: // Default
                    value_out = PreviousDocumentReference;
                    break;
                case 1: // User
                    value_out = "D-" + PreviousDocumentReference;
                    break;
                case 2: // Title for User
                    value_out = "Document ID";
                    break;
                case 3: // Reusable Storage ID (not user visible)
                    value_out = PreviousDocumentReference;
                    break;
                case 4: // = Barcode value
                    value_out = "B_" + PreviousDocumentReference;
                    break;
                default: // Not supported
                    value_out = "";
                    return_code = 0; // Not supported
                    break;
            }

            return return_code;
        }

        int IConnection.EditCustomParameter(string parameter_id, ref UserConfigurableParameter[] connection_parameters, ref string value_out)
        {
            // EzeScan calls this function when the user is about to edit one of the plugin parameters that is of custom type (i.e it uses it's own dialog)
            // parameter_id - the id of the parameter being edited
            // connection_parameters - the current state of the connection parameters
            // value_out - set this parameter to return the new value
            
            // Return values:
            // 0 = User Cancelled
            // 1 = User Accepted
            // less than 0 = error code
            
            // This plugin connection object does not contain any custom parameters - so the return code is always cancel just in case this function is ever called in error
            return 0;
        }

        IOutput IConnection.OutputHandler
        {
            // EzeScan calls this property to retrieve the object that provides the IOutput interface (and therefore handles document output from EzeScan to the Plugin)
            get { return _outputHandler; }
        }        

        IImport IConnection.ImportHandler
        {
            // EzeScan calls this property to retrieve the object that provides the IImport interface (and therefore handles document import from the plugin to EzeScan)

            // Importing of documents is not supported by this sample plugin
    
            get { return null; }
        }        

        IKFI IConnection.KFIHandler
        {
            // EzeScan calls this property to retrieve the object that provides the IKFI interface (and therefore handles KFI field browsing for the plugin)
    
            // KFI Browsing is not supported by this sample plugin
            get { return null; }
        }        

        IUpload IConnection.UploadHandler
        {
            // EzeScan calls this property to retrieve the object that provides the IOutput interface (and therefore handles document and metadata upload from EzeScan to the Plugin)
            
            // Uploading of documents is not supported by this sample plugin
            get { return null; }
        }

        #endregion
    }
}
